<?php
/**
 * Payroll Demo Data Generator
 * Populates the payroll module with sample data for demonstration
 */

defined( 'ABSPATH' ) || exit;

class HRM_Payroll_Demo_Data {
    
    /**
     * Generate complete demo data for payroll module
     */
    public static function generate_demo_data() {
        global $wpdb;
        
        // Check if we already have demo employees
        $existing_employees = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        
        if ($existing_employees == 0) {
            self::create_demo_employees();
        }
        
        // Generate salary structures
        self::create_demo_salary_structures();
        
        // Generate payroll records for last 3 months
        self::create_demo_payroll_records();
        
        // Generate attendance data
        self::create_demo_attendance();
        
        return true;
    }
    
    /**
     * Create demo employees
     */
    private static function create_demo_employees() {
        $employees = [
            ['name' => 'John Anderson', 'email' => 'john.anderson@company.com', 'phone' => '+1-555-0101', 'position' => 'Senior Developer'],
            ['name' => 'Sarah Mitchell', 'email' => 'sarah.mitchell@company.com', 'phone' => '+1-555-0102', 'position' => 'Project Manager'],
            ['name' => 'Michael Chen', 'email' => 'michael.chen@company.com', 'phone' => '+1-555-0103', 'position' => 'UI/UX Designer'],
            ['name' => 'Emily Rodriguez', 'email' => 'emily.rodriguez@company.com', 'phone' => '+1-555-0104', 'position' => 'Marketing Manager'],
            ['name' => 'David Thompson', 'email' => 'david.thompson@company.com', 'phone' => '+1-555-0105', 'position' => 'Sales Executive'],
            ['name' => 'Lisa Wang', 'email' => 'lisa.wang@company.com', 'phone' => '+1-555-0106', 'position' => 'HR Specialist'],
            ['name' => 'James Wilson', 'email' => 'james.wilson@company.com', 'phone' => '+1-555-0107', 'position' => 'Backend Developer'],
            ['name' => 'Maria Garcia', 'email' => 'maria.garcia@company.com', 'phone' => '+1-555-0108', 'position' => 'Content Writer'],
        ];
        
        foreach ($employees as $emp) {
            $employee_id = wp_insert_post([
                'post_title' => $emp['name'],
                'post_type' => 'hrm_employee',
                'post_status' => 'publish',
                'meta_input' => [
                    'email' => $emp['email'],
                    'phone' => $emp['phone'],
                    'position' => $emp['position'],
                    'department' => self::get_random_department(),
                    'joining_date' => date('Y-m-d', strtotime('-' . rand(6, 36) . ' months')),
                ]
            ]);
        }
    }
    
    /**
     * Create demo salary structures
     */
    private static function create_demo_salary_structures() {
        global $wpdb;
        
        $employees = $wpdb->get_results("SELECT ID FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        
        $salary_ranges = [
            ['min' => 45000, 'max' => 65000],
            ['min' => 55000, 'max' => 75000],
            ['min' => 50000, 'max' => 70000],
            ['min' => 60000, 'max' => 80000],
            ['min' => 48000, 'max' => 68000],
        ];
        
        foreach ($employees as $emp) {
            $range = $salary_ranges[array_rand($salary_ranges)];
            $basic_salary = rand($range['min'], $range['max']);
            
            $allowances = json_encode([
                ['label' => 'House Rent Allowance', 'value' => round($basic_salary * 0.40)],
                ['label' => 'Transport Allowance', 'value' => 3000],
                ['label' => 'Medical Allowance', 'value' => 2500],
                ['label' => 'Special Allowance', 'value' => round($basic_salary * 0.10)],
            ]);
            
            $deductions = json_encode([
                ['label' => 'Provident Fund', 'value' => round($basic_salary * 0.12)],
                ['label' => 'Professional Tax', 'value' => 200],
                ['label' => 'Income Tax', 'value' => round($basic_salary * 0.10)],
            ]);
            
            $wpdb->replace(
                $wpdb->prefix . 'hrm_salary_structures',
                [
                    'employee_id' => $emp->ID,
                    'basic_salary' => $basic_salary,
                    'allowances' => $allowances,
                    'deductions' => $deductions,
                    'overtime_rate' => rand(300, 500),
                    'created_at' => date('Y-m-d H:i:s', strtotime('-2 months')),
                    'updated_at' => date('Y-m-d H:i:s'),
                ],
                ['%d', '%f', '%s', '%s', '%f', '%s', '%s']
            );
        }
    }
    
    /**
     * Create demo payroll records
     */
    private static function create_demo_payroll_records() {
        global $wpdb;
        
        $employees = $wpdb->get_results("SELECT ID FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        
        // Generate for last 3 months
        for ($i = 0; $i < 3; $i++) {
            $month = date('Y-m', strtotime("-$i months"));
            
            foreach ($employees as $emp) {
                $structure = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}hrm_salary_structures WHERE employee_id = %d",
                    $emp->ID
                ));
                
                if (!$structure) continue;
                
                $allowances = json_decode($structure->allowances, true);
                $deductions = json_decode($structure->deductions, true);
                
                $total_allowances = array_sum(array_column($allowances, 'value'));
                $total_deductions = array_sum(array_column($deductions, 'value'));
                
                $overtime_hours = rand(0, 15);
                $overtime_pay = $overtime_hours * $structure->overtime_rate;
                
                // Random attendance penalty (0-3 days)
                $absent_days = rand(0, 3);
                $working_days = 22;
                $attendance_penalty = ($structure->basic_salary / $working_days) * $absent_days;
                
                $gross_salary = $structure->basic_salary + $total_allowances + $overtime_pay;
                $net_salary = $gross_salary - $total_deductions - $attendance_penalty;
                
                $status = ($i == 0) ? 'draft' : 'approved';
                
                $payslip_data = json_encode([
                    'employee_name' => get_the_title($emp->ID),
                    'month' => $month,
                    'basic_salary' => $structure->basic_salary,
                    'allowances' => $allowances,
                    'deductions' => $deductions,
                    'overtime_hours' => $overtime_hours,
                    'overtime_rate' => $structure->overtime_rate,
                    'working_days' => $working_days,
                    'present_days' => $working_days - $absent_days,
                    'absent_days' => $absent_days,
                ]);
                
                $wpdb->replace(
                    $wpdb->prefix . 'hrm_payroll',
                    [
                        'employee_id' => $emp->ID,
                        'payroll_month' => $month,
                        'basic_salary' => $structure->basic_salary,
                        'total_allowances' => $total_allowances,
                        'total_deductions' => $total_deductions,
                        'total_bonuses' => 0,
                        'overtime_pay' => $overtime_pay,
                        'attendance_penalty' => $attendance_penalty,
                        'tax_amount' => round($gross_salary * 0.10),
                        'net_salary' => $net_salary,
                        'status' => $status,
                        'payslip_data' => $payslip_data,
                        'created_by' => get_current_user_id(),
                        'created_at' => date('Y-m-d H:i:s', strtotime("$month-25")),
                        'updated_at' => date('Y-m-d H:i:s'),
                    ],
                    ['%d', '%s', '%f', '%f', '%f', '%f', '%f', '%f', '%f', '%f', '%s', '%s', '%d', '%s', '%s']
                );
            }
        }
    }
    
    /**
     * Create demo attendance data
     */
    private static function create_demo_attendance() {
        global $wpdb;
        
        $employees = $wpdb->get_results("SELECT ID FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        
        // Generate attendance for current month
        $start_date = date('Y-m-01');
        $end_date = date('Y-m-d');
        
        $current = strtotime($start_date);
        $end = strtotime($end_date);
        
        while ($current <= $end) {
            $date = date('Y-m-d', $current);
            $day_of_week = date('N', $current);
            
            // Skip weekends
            if ($day_of_week >= 6) {
                $current = strtotime('+1 day', $current);
                continue;
            }
            
            foreach ($employees as $emp) {
                // 95% attendance rate
                $is_present = (rand(1, 100) <= 95);
                
                if ($is_present) {
                    $check_in = date('Y-m-d 09:' . rand(0, 30) . ':00', $current);
                    $check_out = date('Y-m-d 18:' . rand(0, 59) . ':00', $current);
                    $hours = 8 + (rand(0, 120) / 60);
                    
                    $statuses = ['present', 'present', 'present', 'late'];
                    $status = $statuses[array_rand($statuses)];
                } else {
                    $check_in = null;
                    $check_out = null;
                    $hours = 0;
                    $status = 'absent';
                }
                
                $wpdb->replace(
                    $wpdb->prefix . 'hrm_attendance',
                    [
                        'employee_id' => $emp->ID,
                        'date' => $date,
                        'check_in' => $check_in,
                        'check_out' => $check_out,
                        'hours_worked' => $hours,
                        'status' => $status,
                        'created_at' => date('Y-m-d H:i:s'),
                    ],
                    ['%d', '%s', '%s', '%s', '%f', '%s', '%s']
                );
            }
            
            $current = strtotime('+1 day', $current);
        }
    }
    
    /**
     * Get random department
     */
    private static function get_random_department() {
        $departments = ['Engineering', 'Marketing', 'Sales', 'HR', 'Finance', 'Operations'];
        return $departments[array_rand($departments)];
    }
    
    /**
     * Clear all demo data
     */
    public static function clear_demo_data() {
        global $wpdb;
        
        // Get demo employees
        $employees = $wpdb->get_results("SELECT ID FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
        
        foreach ($employees as $emp) {
            wp_delete_post($emp->ID, true);
        }
        
        // Clear tables
        $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}hrm_salary_structures");
        $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}hrm_payroll");
        $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}hrm_attendance");
        
        return true;
    }
}
